/**
  ******************************************************************************
  * @file    startup.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    14-July-2016
  * @brief   Entry point, fault handlers and IVT
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  ******************************************************************************  
  */ 

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The entry point for the application.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */
extern int main(void);

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Reserve space for the system stack. This is the stack that the startup code runs on until the application entry point is
// called. When the boot loader is in standalone mode, the interrupt handler that processes the host communication also uses
// this stack. Make sure that the stack never overflows!
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */

#define BOOTSTACKSIZEINWORDS 256
#define IVTENTRYCT 240

typedef void (*IVTVector)(void);

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The following are variables created by the linker, indicating where the
// "data" and "bss" segments reside in memory.  The initializers for the
// "data" segment resides immediately following the "text" segment.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */
extern unsigned long _sidata;
extern unsigned long _sdata;
extern unsigned long _edata;
extern unsigned long _sbss;
extern unsigned long _ebss;
extern unsigned long _estack;

extern void SysTick_Handler(void);
extern void SystemInit(void);

/** @brief NMI Interrupt Handler
 *
 *  @return Does not return
 */

static void NmiISR(void)
{
    for(;;){}
}

/** @brief Fault Interrupt Handler
 *
 *  This is the hard fault handler. Since we do not enable bus, memory and usage faults,
 *  all those faults will be escalated to this handler (otherwise they would end up in the
 *  default handler because that is where their vector entries point to).
 *
 *  @return Does not return
 */

static void FaultISR(void)
{
    for (;;){}
}

/** @brief Default Handler for all interrupts not handled otherwise
 *
 *  @return Does not return
 */

static  void IntDefaultHandler(void)
{
    for (;;){}
}

/** @brief Reset Interrupt Handler
 *
 *  @return Does not return unless main() returns
 */

static void ResetISR(void)
{
    unsigned    long    *a_Src,*a_Dst;
	  __asm volatile(
					" cpsid i				\n" /* Globally enable interrupts. */);
    // Copy the data segment initializers from flash to SRAM.
    a_Src = &_sidata;
    for(a_Dst = &_sdata; a_Dst < &_edata; )
    {
        *a_Dst++ = *a_Src++;
    }
    // Zero fill the bss segment.
    for(a_Dst = &_sbss; a_Dst < &_ebss; )
    {
        *a_Dst++ = 0;
    }
    SystemInit();
    // Call the application's entry point. An application will frequently call the startup entry point of the C runtime
    // initialization, ie __start which will eventually call main(). But we are in the boot loader here, so we can not
    // afford to have additional code that steals code space and does things we do not have control over.
    main();
}

__attribute__ ((section(".isr_vector")))
IVTVector g_pfnVectorsBL[IVTENTRYCT] =
{
    // this layout of the IVT corresponds to the STM32F4xx family of controllers.
    (IVTVector)(unsigned long)&_estack,                  // initial stack pointer          @0x0000_0000
    ResetISR,                                                  // reset handler                  @0x0000_0004
    NmiISR,                                                    // NMI handler                    @0x0000_0008
    FaultISR,                                                  // hard fault handler             @0x0000_000C
    IntDefaultHandler,                                         // MPU fault handler              @0x0000_0010
    IntDefaultHandler,                                         // bus fault handler              @0x0000_0014
    IntDefaultHandler,                                         // usage fault handler            @0x0000_0018
    0,                                                            // Reserved                       @0x0000_001C
    0,                                                            // Reserved                       @0x0000_0020
    0,                                                            // Reserved                       @0x0000_0024
    0,                                                            // Reserved                       @0x0000_0028
    IntDefaultHandler,                                         // SVCall handler                 @0x0000_002C
    IntDefaultHandler,                                         // Debug monitor handler          @0x0000_0030
    0,                                                            // Reserved                       @0x0000_0034
    IntDefaultHandler,                                         // PendSV handler                 @0x0000_0038
    SysTick_Handler,                                           // SysTick handler                @0x0000_003C
    /* External Interrupts */
    IntDefaultHandler,                                         // WWDG_IRQHandler                @0x0000_0040                                    
    IntDefaultHandler,                                         // PVD_IRQHandler                 @0x0000_0044            
    IntDefaultHandler,                                         // TAMP_STAMP_IRQHandler          @0x0000_0048            
    IntDefaultHandler,                                         // RTC_WKUP_IRQHandler            @0x0000_004C           
    IntDefaultHandler,                                         // FLASH_IRQHandler               @0x0000_0050                           
    IntDefaultHandler,                                         // RCC_IRQHandler                 @0x0000_0054                             
    IntDefaultHandler,                                         // EXTI0_IRQHandler               @0x0000_0058         
    IntDefaultHandler,                                         //                                @0x0000_005C
    IntDefaultHandler,                                         // EXTI2_IRQHandler               @0x0000_0060           
    IntDefaultHandler,                                         // EXTI3_IRQHandler               @0x0000_0064           
    IntDefaultHandler,                                         // EXTI4_IRQHandler               @0x0000_0068           
    IntDefaultHandler,                                         // DMA1_Stream0_IRQHandler        @0x0000_006C   
    IntDefaultHandler,                                         // DMA1_Stream1_IRQHandler        @0x0000_0070    
    IntDefaultHandler,                                         // DMA1_Stream2_IRQHandler        @0x0000_0074    
    IntDefaultHandler,                                         // DMA1_Stream3_IRQHandler        @0x0000_0078    
    IntDefaultHandler,                                         // DMA1_Stream4_IRQHandler        @0x0000_007C    
    IntDefaultHandler,                                         // DMA1_Stream5_IRQHandler        @0x0000_0080    
    IntDefaultHandler,                                         // DMA1_Stream6_IRQHandler        @0x0000_0084    
    IntDefaultHandler,                                         // ADC_IRQHandler                 @0x0000_0088    
    IntDefaultHandler,                                         // CAN1_TX_IRQHandler             @0x0000_008C          
    IntDefaultHandler,                                         // CAN1_RX0_IRQHandler            @0x0000_0090           
    IntDefaultHandler,                                         // CAN1_RX1_IRQHandler            @0x0000_0094           
    IntDefaultHandler,                                         // CAN1_SCE_IRQHandler            @0x0000_0098           
    IntDefaultHandler,                                         // EXTI9_5_IRQHandler             @0x0000_009C           
    IntDefaultHandler,                                         // TIM1_BRK_TIM9_IRQHandler       @0x0000_00A0 
    IntDefaultHandler,                                         // TIM1_UP_TIM10_IRQHandler       @0x0000_00A4 
    IntDefaultHandler,                                         // TIM1_TRG_COM_TIM11_IRQHandler  @0x0000_00A8 
    IntDefaultHandler,                                         // TIM1_CC_IRQHandler             @0x0000_00AC           
    IntDefaultHandler,                                         // TIM2_IRQHandler                @0x0000_00B0    
    IntDefaultHandler,                                         // TIM3_IRQHandler                @0x0000_00B4    
    IntDefaultHandler,                                         // TIM4_IRQHandler                @0x0000_00B8    
    IntDefaultHandler,                                         // I2C1_EV_IRQHandler             @0x0000_00BC           
    IntDefaultHandler,                                         // I2C1_ER_IRQHandler             @0x0000_00C0           
    IntDefaultHandler,                                         // I2C2_EV_IRQHandler             @0x0000_00C4           
    IntDefaultHandler,                                         // I2C2_ER_IRQHandler             @0x0000_00C8             
    IntDefaultHandler,                                         // SPI1_IRQHandler                @0x0000_00CC    
    IntDefaultHandler,                                         // SPI2_IRQHandler                @0x0000_00D0    
    IntDefaultHandler,                                         // USART1_IRQHandler              @0x0000_00D4    
    IntDefaultHandler,                                         // USART2_IRQHandler              @0x0000_00D8    
    IntDefaultHandler,                                         // USART3_IRQHandler              @0x0000_00DC    
    IntDefaultHandler,                                         // EXTI15_10_IRQHandler           @0x0000_00E0           
    IntDefaultHandler,                                         // RTC_Alarm_IRQHandler           @0x0000_00E4           
    IntDefaultHandler,                                         // OTG_FS_WKUP_IRQHandler         @0x0000_00E8               
    IntDefaultHandler,                                         // TIM8_BRK_TIM12_IRQHandler      @0x0000_00EC 
    IntDefaultHandler,                                         // TIM8_UP_TIM13_IRQHandler       @0x0000_00F0 
    IntDefaultHandler,                                         // TIM8_TRG_COM_TIM14_IRQHandler  @0x0000_00F4 
    IntDefaultHandler,                                         // TIM8_CC_IRQHandler             @0x0000_00F8           
    IntDefaultHandler,                                         // DMA1_Stream7_IRQHandler        @0x0000_00FC           
    IntDefaultHandler,                                         // FSMC_IRQHandler                @0x0000_0100    
    IntDefaultHandler,                                         // SDIO_IRQHandler                @0x0000_0104    
    IntDefaultHandler,                                         // TIM5_IRQHandler                @0x0000_0108    
    IntDefaultHandler,                                         // SPI3_IRQHandler                @0x0000_010C    
    IntDefaultHandler,                                         // UART4_IRQHandler               @0x0000_0110    
    IntDefaultHandler,                                         // UART5_IRQHandler               @0x0000_0114    
    IntDefaultHandler,                                         // TIM6_DAC_IRQHandler            @0x0000_0118       
    IntDefaultHandler,                                         // TIM7_IRQHandler                @0x0000_011C 
    IntDefaultHandler,                                         // DMA2_Stream0_IRQHandler        @0x0000_0120    
    IntDefaultHandler,                                         // DMA2_Stream1_IRQHandler        @0x0000_0124    
    IntDefaultHandler,                                         // DMA2_Stream2_IRQHandler        @0x0000_0128    
    IntDefaultHandler,                                         // DMA2_Stream3_IRQHandler        @0x0000_012C    
    IntDefaultHandler,                                         // DMA2_Stream4_IRQHandler        @0x0000_0130    
    IntDefaultHandler,                                         // ETH_IRQHandler                 @0x0000_0134    
    IntDefaultHandler,                                         // ETH_WKUP_IRQHandler            @0x0000_0138           
    IntDefaultHandler,                                         // CAN2_TX_IRQHandler             @0x0000_013C           
    IntDefaultHandler,                                         // CAN2_RX0_IRQHandler            @0x0000_0140           
    IntDefaultHandler,                                         // CAN2_RX1_IRQHandler            @0x0000_0144           
    IntDefaultHandler,                                         // CAN2_SCE_IRQHandler            @0x0000_0148           
    IntDefaultHandler,                                         // OTG_FS_IRQHandler              @0x0000_014C    
    IntDefaultHandler,                                         // DMA2_Stream5_IRQHandler        @0x0000_0150    
    IntDefaultHandler,                                         // DMA2_Stream6_IRQHandler        @0x0000_0154    
    IntDefaultHandler,                                         // DMA2_Stream7_IRQHandler        @0x0000_0158    
    IntDefaultHandler,                                         // USART6_IRQHandler              @0x0000_015C     
    IntDefaultHandler,                                         // I2C3_EV_IRQHandler             @0x0000_0160           
    IntDefaultHandler,                                         // I2C3_ER_IRQHandler             @0x0000_0164           
    IntDefaultHandler,                                         // OTG_HS_EP1_OUT_IRQHandler      @0x0000_0168    
    IntDefaultHandler,                                         // OTG_HS_EP1_IN_IRQHandler       @0x0000_016C    
    IntDefaultHandler,                                         // OTG_HS_WKUP_IRQHandler         @0x0000_0170            
    IntDefaultHandler,                                         // OTG_HS_IRQHandler              @0x0000_0174    
    IntDefaultHandler,                                         // DCMI_IRQHandler                @0x0000_0178    
    IntDefaultHandler,                                         // CRYP_IRQHandler                @0x0000_017C    
    IntDefaultHandler,                                         // HASH_RNG_IRQHandler            @0x0000_0180 
    IntDefaultHandler,                                         // FPU_IRQHandler                 @0x0000_0184    
    IntDefaultHandler,                                         // UART7_IRQHandler               @0x0000_0188           
    IntDefaultHandler,                                         // UART8_IRQHandler               @0x0000_018C    
    IntDefaultHandler,                                         // SPI4_IRQHandler                @0x0000_0190    
    IntDefaultHandler,                                         // SPI5_IRQHandler                @0x0000_0194            
    IntDefaultHandler,                                         // SPI6_IRQHandler                @0x0000_0198    
    IntDefaultHandler,                                         // SAI1_IRQHandler                @0x0000_019C    
    IntDefaultHandler,                                         // LCD_TFTIRQHandler              @0x0000_01A0    
    IntDefaultHandler,                                         // LCD_TFTIRQErrorHandler         @0x0000_01A4 
    IntDefaultHandler,                                         // DMA2D_IRQHandler               @0x0000_01A8          
};    

